// dashboard/app.js - Versión mejorada con APIs

let currentLocation = 'Cajabamba';
let currentConductorType = 'privado';

// Función para cargar conductores desde la API
async function loadConductores(tipo) {
    try {
        showLoading(tipo);
        
        const response = await fetch(`${apiBaseUrl}/get_conductores.php?tipo=${tipo}&ubicacion=${currentLocation}`);
        const data = await response.json();
        
        if (data.success) {
            displayConductores(data.conductores, tipo);
        } else {
            showError('Error al cargar conductores: ' + data.error);
        }
    } catch (error) {
        console.error('Error:', error);
        showError('Error de conexión');
    }
}

// Mostrar loading
function showLoading(tipo) {
    const container = tipo === 'privado' ? 
        document.getElementById('candado') : 
        document.getElementById('publica');
    
    container.innerHTML = `
        <div class="loandig">
            <span></span>
            <span></span>
            <span></span>
            <span></span>
        </div>
    `;
}

// Mostrar error
function showError(message) {
    const container = document.getElementById(currentConductorType === 'privado' ? 'candado' : 'publica');
    container.innerHTML = `
        <div class="card">
            <div style="text-align: center; padding: 40px 20px; color: #666;">
                <ion-icon name="warning-outline" style="font-size: 48px; color: #ff9800;"></ion-icon>
                <h3>Error</h3>
                <p>${message}</p>
                <button onclick="loadConductores('${currentConductorType}')" 
                        style="margin-top: 15px; padding: 10px 20px; background: #37B3FF; color: white; border: none; border-radius: 5px;">
                    Reintentar
                </button>
            </div>
        </div>
    `;
}

// Mostrar conductores en el panel
function displayConductores(conductores, tipo) {
    const container = tipo === 'privado' ? 
        document.getElementById('candado') : 
        document.getElementById('publica');
    
    if (conductores.length === 0) {
        container.innerHTML = `
            <div class="card">
                <div style="text-align: center; padding: 40px 20px; color: #666;">
                    <ion-icon name="car-outline" style="font-size: 48px;"></ion-icon>
                    <h3>No hay conductores disponibles</h3>
                    <p>Intenta más tarde o cambia de ubicación</p>
                </div>
            </div>
        `;
        return;
    }

    let html = '';
    
    conductores.forEach(conductor => {
        if (tipo === 'privado') {
            html += generatePrivateConductorHTML(conductor);
        } else {
            html += generateGroupConductorHTML(conductor);
        }
    });
    
    container.innerHTML = html;
}

// Generar HTML para conductor privado
function generatePrivateConductorHTML(conductor) {
    const metodosPago = conductor.metodo_pago || [];
    
    return `
        <div class="card conductor">
            <div class="conductor_nav">
                <div class="conductor_nav_left">
                    <img src="../assets/img/icon.png" alt="logo">
                </div>
                <div class="conductor_nav_right">
                    <h3>${conductor.nombre} <img src="https://cdn-icons-png.flaticon.com/128/5362/5362616.png"></h3>
                    <div style="display: flex; align-items: center;">
                        <p>${conductor.conductor_estado === 'disponible' ? 'Activo' : 'Ocupado'}</p>
                        <p style="background: #FF2F2F95; margin: 0 3px">${conductor.ubicacion_actual}</p>
                    </div>
                    <div class="rating">
                        <div class="rating_star">
                            ${generateStars(conductor.rating)}
                        </div>
                        <h2>${conductor.rating} (${conductor.total_viajes} reseñas)</h2>
                    </div>
                </div>
                <div class="conductor_nav_activity">
                    <ion-icon name="ellipsis-vertical"></ion-icon>
                    <div class="tooltip">
                        <div class="tooltip_option">
                            <ion-icon name="person"></ion-icon>
                            <p>Reportar conductor</p>
                        </div>
                        <div class="tooltip_option">
                            <ion-icon name="share-social"></ion-icon>
                            <p>Compartir</p>
                        </div>
                        <div class="tooltip_option">
                            <ion-icon name="star"></ion-icon>
                            <p>Reseñas</p>
                        </div>
                        <div class="tooltip_option">
                            <ion-icon name="document"></ion-icon>
                            <p>Detalles de auto</p>
                        </div>
                    </div>
                </div>
            </div>
            <br>
            <div class="conductor_stactics">
                <div class="conductor_stactics_data">
                    <ion-icon name="car-sport"></ion-icon>
                    <div class="">
                        <h3>Viajes</h3>
                        <p>${conductor.total_viajes}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="checkmark-circle"></ion-icon>
                    <div class="">
                        <h3>Exitosos</h3>
                        <p>${conductor.viajes_exitosos || 0}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="chevron-collapse"></ion-icon>
                    <div class="">
                        <h3>Fallo</h3>
                        <p>${conductor.viajes_fallidos || 0}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="warning"></ion-icon>
                    <div class="">
                        <h3>Reportes</h3>
                        <p>${conductor.reportes || 0}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="caret-forward-circle"></ion-icon>
                    <div class="">
                        <h3>Km</h3>
                        <p>${conductor.kilometros_recorridos || 0}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="alarm"></ion-icon>
                    <div class="">
                        <h3>Horas</h3>
                        <p>${conductor.horas_conduccion || 0}</p>
                    </div>
                </div>
            </div>
            <div class="conductor_price">
                <p>Rango de precios</p>
                <h3>S/. ${conductor.precio_min} - S/. ${conductor.precio_max}</h3>
                <div class="scrollview_data">
                    <p>Billeteras de pago aceptadas</p>
                    <div class="scrollview_data_dn">
                        ${generatePaymentMethods(metodosPago)}
                    </div>
                </div>
            </div>
            <div class="conductor_contract">
                <div class="conductor_btn" onclick="contratarViaje(${conductor.id})">
                    <ion-icon name=""></ion-icon>
                    <p>Contratar Viaje</p>
                </div>
                <div class="btn_call" onclick="llamarConductor('${conductor.telefono}')">
                    <ion-icon name="call"></ion-icon>
                </div>
                <div class="btn_call" onclick="whatsappConductor('${conductor.telefono}')">
                    <ion-icon name="logo-WhatsApp"></ion-icon>
                </div>
            </div>
        </div>
    `;
}

// Generar HTML para conductor grupal
function generateGroupConductorHTML(conductor) {
    return `
        <div class="card publica">
            <div class="conductor_nav">
                <div class="conductor_nav_left">
                    <img src="../assets/img/icon.png" alt="icon">
                </div>
                <div class="conductor_nav_right">
                    <h3>${conductor.nombre}</h3>
                    <div style="display: flex; align-items: center;">
                        <p>Conduciendo</p>
                        <p style="background: #37B3FF95; margin: 0 3px">${conductor.ubicacion_actual}</p>
                    </div>
                    <div class="rating">
                        <div class="tating_star">
                            ${generateStars(conductor.rating)}
                        </div>
                        <h2>${conductor.rating} (${conductor.total_viajes} reseñas)</h2>
                    </div>
                </div>
            </div>
            </br>
            <div class="conductor_stactics">
                <div class="conductor_stactics_data">
                    <ion-icon name="checkmark-done-circle"></ion-icon>
                    <div class="">
                        <h3>Tasa(éxito)</h3>
                        <p>${calculateSuccessRate(conductor)}%</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="person-add"></ion-icon>
                    <div class="">
                        <h3>Lugares(D)</h3>
                        <p>3</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="person-remove"></ion-icon>
                    <div class="">
                        <h3>Lugares(O)</h3>
                        <p>5</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="warning"></ion-icon>
                    <div class="">
                        <h3>Reportes</h3>
                        <p>${conductor.reportes || 0}</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="caret-forward-circle"></ion-icon>
                    <div class="">
                        <h3>Km(D)</h3>
                        <p>${conductor.kilometros_recorridos || 0}km</p>
                    </div>
                </div>
                <div class="conductor_stactics_data">
                    <ion-icon name="alarm"></ion-icon>
                    <div class="">
                        <h3>Horas(es)</h3>
                        <p>${conductor.horas_conduccion || 0}m</p>
                    </div>
                </div>
            </div>
            <div class="conductor_price">
                <p>Rango de precios</p>
                <h3>S/. ${conductor.precio_min} - S/. ${conductor.precio_max}</h3>
                <div class="scrollview_data">
                    <p>Billeteras de pago aceptadas</p>
                    <div class="scrollview_data_dn">
                        ${generatePaymentMethods(conductor.metodo_pago || [])}
                    </div>
                </div>
            </div>
            <div class="conductor_contract">
                <div class="conductor_btn" onclick="enviarNotificacion()">
                    <ion-icon name=""></ion-icon>
                    <p>Unirme al viaje</p>
                </div>
                <div class="btn_call" onclick="llamarConductor('${conductor.telefono}')">
                    <ion-icon name="call"></ion-icon>
                </div>
                <div class="btn_call" onclick="whatsappConductor('${conductor.telefono}')">
                    <ion-icon name="logo-WhatsApp"></ion-icon>
                </div>
            </div>
        </div>
    `;
}

// Calcular tasa de éxito
function calculateSuccessRate(conductor) {
    const total = (conductor.viajes_exitosos || 0) + (conductor.viajes_fallidos || 0);
    if (total === 0) return 0;
    return Math.round((conductor.viajes_exitosos / total) * 100);
}

// Generar estrellas basado en rating
function generateStars(rating) {
    let stars = '';
    const fullStars = Math.floor(rating);
    const hasHalfStar = rating % 1 >= 0.5;
    
    for (let i = 0; i < fullStars; i++) {
        stars += '<ion-icon name="star"></ion-icon>';
    }
    
    if (hasHalfStar) {
        stars += '<ion-icon name="star-half"></ion-icon>';
    }
    
    const emptyStars = 5 - fullStars - (hasHalfStar ? 1 : 0);
    for (let i = 0; i < emptyStars; i++) {
        stars += '<ion-icon name="star-outline"></ion-icon>';
    }
    
    return stars;
}

// Generar métodos de pago
function generatePaymentMethods(metodos) {
    const paymentIcons = {
        'yape': '../assets/img/yape-37283.png',
        'plin': '../assets/img/plin-interbank-4391.png',
        'efectivo': '../assets/img/cash.jpeg',
        'transferencia': 'https://cdn-icons-png.flaticon.com/512/18209/18209624.png'
    };
    
    let html = '';
    metodos.forEach(metodo => {
        if (paymentIcons[metodo]) {
            html += `<img src="${paymentIcons[metodo]}" alt="${metodo}">`;
        }
    });
    
    return html;
}

// Funciones para tipos de viaje
function tipePrivate() {
    currentConductorType = 'privado';
    var a = document.querySelector(".indicator_btn");
    var div = document.getElementById("candado");
    a.style.transform = "translateX(0px)";
    document.getElementById("publica").innerHTML = ''; 
    
    loadConductores('privado');
}

function tipeGrupo() {
    currentConductorType = 'grupo';
    var a = document.querySelector(".indicator_btn");
    var div = document.getElementById("publica");
    div.style.display = "";
    a.style.transform = "translateX(90px)";
    document.getElementById("candado").innerHTML = ''; 
    
    loadConductores('grupo');
}

// Funciones de contacto
function llamarConductor(telefono) {
    window.open(`tel:${telefono}`, '_self');
}

function whatsappConductor(telefono) {
    const mensaje = encodeURIComponent('Hola, estoy interesado en contratar un viaje con TreyNov');
    window.open(`https://wa.me/${telefono}?text=${mensaje}`, '_blank');
}

function contratarViaje(conductorId) {
    alert(`Contratando viaje con conductor ID: ${conductorId}`);
    // Aquí iría la lógica real de contratación
}

function enviarNotificacion() {
    alert('Solicitud de unión al viaje enviada');
}

// Cargar notificaciones
async function loadNotifications() {
    try {
        const response = await fetch(`${apiBaseUrl}/notifications.php?action=list`);
        const data = await response.json();
        
        if (data.success) {
            displayNotifications(data.notifications);
        }
    } catch (error) {
        console.error('Error cargando notificaciones:', error);
    }
}

function displayNotifications(notifications) {
    const container = document.getElementById('notificationsList');
    
    if (notifications.length === 0) {
        container.innerHTML = `
            <div style="text-align: center; padding: 40px 20px; color: #666;">
                <ion-icon name="notifications-off-outline" style="font-size: 48px;"></ion-icon>
                <h3>No hay notificaciones</h3>
                <p>No tienes notificaciones en este momento</p>
            </div>
        `;
        return;
    }
    
    let html = '';
    notifications.forEach(notif => {
        html += `
            <div class="new_notifi_card" onclick="markNotificationAsRead(${notif.id})" 
                 style="${notif.leida ? 'opacity: 0.7;' : ''}">
                <ion-icon name="notifications" style="${getNotificationIcon(notif.tipo)}"></ion-icon>
                <div class="">
                    <h4>${notif.titulo}</h4>
                    <p>${notif.mensaje}</p>
                    <b>${formatDate(notif.fecha_creacion)}</b>
                </div>
            </div>
        `;
    });
    
    container.innerHTML = html;
}

function getNotificationIcon(type) {
    const icons = {
        'info': 'color: #2196F3;',
        'success': 'color: #4CAF50;',
        'warning': 'color: #FF9800;',
        'error': 'color: #F44336;'
    };
    return icons[type] || 'color: #2196F3;';
}

function formatDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diff = now - date;
    
    if (diff < 60000) return 'Ahora';
    if (diff < 3600000) return `${Math.floor(diff / 60000)} min ago`;
    if (diff < 86400000) return `${Math.floor(diff / 3600000)} h ago`;
    return date.toLocaleDateString();
}

async function markNotificationAsRead(notificationId) {
    try {
        const response = await fetch(`${apiBaseUrl}/notifications.php`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'mark_read',
                notification_id: notificationId
            })
        });
        
        const data = await response.json();
        if (data.success) {
            loadNotifications(); // Recargar notificaciones
        }
    } catch (error) {
        console.error('Error marcando notificación:', error);
    }
}

// Chat con soporte
function sendChatMessage() {
    const input = document.getElementById('chatInput');
    const message = input.value.trim();
    
    if (message === '') return;
    
    const chatBox = document.querySelector('.soporte_bot .chat_online');
    const bubble = document.createElement('div');
    bubble.className = 'chat_online_cliente';
    bubble.innerHTML = `
        <h4><div class="bulb_online"></div>Tu</h4>
        <p>${message}</p>
        <br>
        <span>${new Date().toLocaleDateString()}</span>
    `;
    
    chatBox.appendChild(bubble);
    chatBox.scrollTop = chatBox.scrollHeight;
    input.value = '';
    
    // Simular respuesta del bot
    setTimeout(() => {
        const response = document.createElement('div');
        response.className = 'chat_online_person';
        response.innerHTML = `
            <h4>Aymel (IA)<div class="bulb_online"></div></h4>
            <p>He recibido tu mensaje: "${message}". ¿En qué más puedo ayudarte?</p>
            <br>
            <span>${new Date().toLocaleDateString()}</span>
        `;
        chatBox.appendChild(response);
        chatBox.scrollTop = chatBox.scrollHeight;
    }, 1000);
}

// Actualización en tiempo real
setInterval(() => {
    loadConductores(currentConductorType);
}, 30000); // Actualizar cada 30 segundos

// Actualizar datos del usuario
setInterval(updateUserData, 60000);

async function updateUserData() {
    try {
        const response = await fetch(`${apiBaseUrl}/get_user_data.php`);
        const data = await response.json();
        
        if (data.success) {
            // Actualizar UI con nuevos datos
            document.querySelector('.statics_data:nth-child(1) p').textContent = data.user.total_viajes;
            document.querySelector('.statics_data:nth-child(2) p').textContent = data.user.total_contactos + '/10';
            document.querySelector('.statics_data:nth-child(3) p').textContent = 'S/.' + data.user.saldo.toFixed(2);
            document.querySelector('.balance-display').textContent = 'S/. ' + data.user.saldo.toFixed(2);
            
            // Actualizar sesión
            Object.keys(data.user).forEach(key => {
                if (key.startsWith('user_')) {
                    sessionStorage.setItem(key, data.user[key]);
                }
            });
        }
    } catch (error) {
        console.error('Error updating user data:', error);
    }
}

// Inicializar
document.addEventListener('DOMContentLoaded', function() {
    tipePrivate(); // Cargar conductores privados por defecto
    loadNotifications(); // Cargar notificaciones
    
    // Configurar toggles
    initializeToggles();
});

// Logout
function logout() {
    if (confirm('¿Estás seguro de que quieres cerrar sesión?')) {
        window.location.href = '../logout.php';
    }
}

// Inicializar toggles de configuración
function initializeToggles() {
    const toggles = ['notif', 'alerts', 'promotion', 'factory', 'vf', 'yape', 'plin', 'efective', 'bank'];
    
    toggles.forEach(toggleId => {
        const toggle = document.getElementById(`t-${toggleId}`);
        if (toggle) {
            // Cargar estado guardado
            const savedState = localStorage.getItem(`toggle_${toggleId}`);
            if (savedState === 'true') {
                toggle.classList.add('on');
                toggle.classList.remove('off');
            }
            
            toggle.addEventListener('click', function() {
                this.classList.toggle('on');
                this.classList.toggle('off');
                localStorage.setItem(`toggle_${toggleId}`, this.classList.contains('on'));
            });
        }
    });
}